/***************************************************************************
 *
 * Copyright (c) 2014 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdint.h>
#include <stdbool.h>

enum e_pixel_format
{
    PIXEL_FORMAT_INVALID = 0,
    PIXEL_FORMAT_8888 = 1,
    PIXEL_FORMAT_565 = 2
};

int formatToBytes(enum e_pixel_format format)
{
    switch (format)
    {
        case PIXEL_FORMAT_8888:
            return 4;
        case PIXEL_FORMAT_565:
            return 2;
        default:
            fprintf(stderr, "Invalid pixel format '%d'\n", format);
            return 0;
    }
}

int diff(FILE* file1, FILE* file2, enum e_pixel_format format, int tolerance, FILE* diff_output)
{
    int pixelSize = formatToBytes(format);

    bool anyDifferent = false;

    unsigned long pixelIndex = 0;
    if (pixelSize == 0)
    {
        return EXIT_FAILURE;
    }

    while (true)
    {
        int file1_end, file2_end;
        uint32_t file1_data = 0, file2_data = 0;

        fread(&file1_data, pixelSize, 1, file1);

        fread(&file2_data, pixelSize, 1, file2);

        file1_end = feof(file1);
        file2_end = feof(file2);
        if (file1_end != 0 || file2_end != 0)
        {
            if (file1_end != 0 && file2_end != 0)
            {
                if (anyDifferent == true)
                {
                    return EXIT_FAILURE;
                }
                else
                {
                    return EXIT_SUCCESS;
                }
            }
            else
            {
                if (file1_end != 0)
                    fprintf(stderr, "FILE_1 ended before FILE_2\n");
                else if (file2_end != 0)
                    fprintf(stderr, "FILE_2 ended before FILE_1\n");
                else
                    fprintf(stderr, "This should not happen\n");

                return EXIT_FAILURE;
            }
        }

        if (ferror(file1) != 0)
        {
            fprintf(stderr, "FILE_1 had an error\n");
            return EXIT_FAILURE;
        }
        if (ferror(file2) != 0)
        {
            fprintf(stderr, "FILE_2 had an error\n");
            return EXIT_FAILURE;
        }

        if (format == PIXEL_FORMAT_8888)
        {
            uint32_t mask, val1, val2;
            int i;
            bool isDifferent = false;
            for (i = 0; i < 4; i++)
            {
                mask = 0xFF << (i * 8);
                val1 = (file1_data & mask) >> (i * 8);
                val2 = (file2_data & mask) >> (i * 8);
                if (abs(val2 - val1) > tolerance)
                {
                    isDifferent = true;
                    anyDifferent = true;
                }
            }
            if (isDifferent)
            {
                fprintf(diff_output, "Difference detected in pixel %lu "
                        "(0x%08x vs. 0x%08x)\n",
                        pixelIndex, file1_data, file2_data);
            }
        }
        else if (format == PIXEL_FORMAT_565)
        {
            uint32_t mask, val1, val2;
            bool isDifferent = false;

            if (file1_data & 0xFFFF0000)
            {
                fprintf(stderr, "ERROR! file1_data is using the upper 16 bits (0x%08x)\n", file1_data);
                exit(EXIT_FAILURE);
            }
            if (file2_data & 0xFFFF0000)
            {
                fprintf(stderr, "ERROR! file2_data is using the upper 16 bits (0x%08x)\n", file2_data);
                exit(EXIT_FAILURE);
            }

            // XX5
            mask = 0x1F;
            val1 = (file1_data & mask);
            val2 = (file2_data & mask);
            if (abs(val2 - val1) > tolerance)
            {
                isDifferent = true;
                anyDifferent = true;
            }

            // X6X
            mask = 0x3F << 5;
            val1 = (file1_data & mask) >> 5;
            val2 = (file2_data & mask) >> 5;
            if (abs(val2 - val1) > tolerance)
            {
                isDifferent = true;
                anyDifferent = true;
            }

            // 5XX
            mask = 0x1F << 11;
            val1 = (file1_data & mask) >> 11;
            val2 = (file2_data & mask) >> 11;
            if (abs(val2 - val1) > tolerance)
            {
                isDifferent = true;
                anyDifferent = true;
            }

            if (isDifferent)
            {
                fprintf(diff_output, "Difference detected in pixel %lu "
                        "(0x%04x vs. 0x%04x)\n",
                        pixelIndex, file1_data, file2_data);
            }
        }
        else
        {
            fprintf(stderr, "Unknown pixel format %d\n", format);
            return EXIT_FAILURE;
        }

        pixelIndex++;

    }

}

void usage(char *prog)
{
    printf("Usage: %s FILE_1 FILE_2 FORMAT TOLERANCE [DIFF_OUTPUT]\n"
           "  where FORMAT may be '565' or '8888'\n"
           "  and TOLERANCE is an integer between 0 and 255\n", prog);
    exit(EXIT_FAILURE);
}

int main(int argc, char** argv)
{
    FILE *file1 = NULL;
    FILE *file2 = NULL;
    FILE *output_file = stdout;
    enum e_pixel_format format = PIXEL_FORMAT_INVALID;
    char *end;
    int tolerance;
    int retval;

    if (argc < 5)
        usage(argv[0]);

    file1 = fopen(argv[1], "r");
    if (file1 == NULL)
    {
        fprintf(stderr, "Failed to open '%s'\n", argv[1]);
        usage(argv[0]);
    }

    file2 = fopen(argv[2], "r");
    if (file2 == NULL)
    {
        fprintf(stderr, "Failed to open '%s'\n", argv[2]);
        usage(argv[0]);
    }

    if (strncmp(argv[3], "565", sizeof("565")) == 0)
    {
        format = PIXEL_FORMAT_565;
    }
    else if (strncmp(argv[3], "8888", sizeof("565")) == 0)
    {
        format = PIXEL_FORMAT_8888;
    }
    else
    {
        fprintf(stderr, "Unrecognized pixel format '%s'\n", argv[3]);
        usage(argv[0]);
    }

    if (*argv[4] == '\0')
    {
        fprintf(stderr, "TOLERANCE is an empty string\n");
        usage(argv[0]);
    }
    tolerance = strtol(argv[4], &end, 10);
    if (*end != '\0')
    {
        fprintf(stderr, "Failed to read TOLERANCE\n");
        usage(argv[0]);
    }

    if (tolerance < 0 || tolerance > 255)
    {
        fprintf(stderr, "TOLERANCE of %d is outside the expected range\n",
                tolerance);
        usage(argv[0]);
    }

    if (argc >= 6)
    {
        if (strcmp(argv[5], "-") == 0)
        {
            output_file = stdout;
        }
        else
        {
            output_file = fopen(argv[5], "w");
            if (output_file == NULL)
            {
                fprintf(stderr, "Failed to open '%s' for writing", argv[5]);
                usage(argv[0]);
            }
        }
    }

    retval = diff(file1, file2, format, tolerance, output_file);
    if (output_file != stdout)
    {
        fclose(output_file);
    }
    return retval;
}
